/*
 * CollisionObject.h
 *
 * Created 8/17/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/17/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef COLLISION_OBJECT_H
 #define COLLISION_OBJECT_H
 
 template <typename T> class CollisionObject;
 
 #include "Object.h"
 
 // For collision handling
 #include "CollisionHandlerEventNodeCollection.h"
 
 #include "global.h"
 
 #include "StringConversion.h"
 
 /**
  * Class specification for CollisionObject
  */
 template <typename T>
 class CollisionObject : virtual public Object<T>
 {
 // Data Members
 private:
    ColliderCollection _colliders;
 
 // Local Functions
 public:
    CollisionObject();
    CollisionObject( const CollisionObject<T>& col_obj );
    virtual ~CollisionObject();
    inline CollisionObject<T>& operator=( const CollisionObject<T>& col_obj );
    virtual inline void add_collider( CollisionHandlerEventNode<T>* collider_Ptr );
    virtual inline void add_collider( const std::string& collider_name, CollisionHandlerEventNode<T>* collider_Ptr );
    virtual inline void detach_collider( const std::string& key );
    virtual inline void detach_colliders();
    virtual inline CollisionHandlerEventNode<T>* get_collider( const std::string& key );
    //virtual inline const ColliderCollection& get_colliders() const;
    virtual inline CollisionHandlerEventNode<T>* get_first_active_collider_found();
    virtual inline void remove_collider( const std::string& key );
    virtual inline void remove_colliders();
    virtual inline void reparent_collider_to( const std::string& key, NodePath& node_path );
    virtual inline void reparent_colliders_to( NodePath& node_path );
    virtual inline void tag_colliders( const std::string& tag_key, const std::string& tag_value );
    virtual inline void turn_on_collision();
    virtual inline void turn_off_collision();
    
    // overloaded functions (NodePath)
    /*static void init_type() {
                                std::string template_type( typeid( T ).name() );
                                register_type(_type_handle, "CollisionObject<" + template_type + ">" );
                            }*/
 
 // Protected Functions
 protected:
    virtual inline void add_collider_without_tagging( const std::string& collider_name, CollisionHandlerEventNode<T>* collider_Ptr );
 
 // Private Functions
 private:
 
 // Friend Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 CollisionObject<T>::CollisionObject()
                    : Object<T>()
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 CollisionObject<T>::CollisionObject( const CollisionObject<T>& col_obj )
                    : Object<T>( static_cast< Object<T> >( col_obj ) )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 CollisionObject<T>::~CollisionObject()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified CollisionObject to this CollisionObject.
  *
  * @param (const CollisionObject<T>& col_obj )
  * @return CollisionObject<T>&
  */
 template <typename T>
 inline CollisionObject<T>& CollisionObject<T>::operator=( const CollisionObject<T>& col_obj )
 {
    Object<T>::operator=( static_cast< Object<T> >( col_obj ) );
    
    return *this;
 }
 
 /**
  * get_colliders() return a reference to the ColliderCollection of this Object.
  *
  * @return const ColliderCollection&
  */
 //template <typename T>
 //inline const ColliderCollection& CollisionObject<T>::get_colliders() const
 //{
 //   return _colliders;
 //}
 
 /**
  * add_collider() adds the Collider, specified by the collider_Ptr, using its name as the key, 
  * into the ColliderCollection for this Object.
  *
  * @param (CollisionHandlerEventNode<T>*) collider_Ptr
  */
 template <typename T>
 inline void CollisionObject<T>::add_collider( CollisionHandlerEventNode<T>* collider_Ptr )
 {
    nassertv( collider_Ptr != NULL );
    nassertv( !collider_Ptr->is_empty() );
    CollisionObject<T>::add_collider( collider_Ptr->get_name(), collider_Ptr );
 }
 
 /**
  * add_collider() adds the Collider specified by the collider_Ptr with the specified key into the
  * ColliderCollection for this Object.
  *
  * @param (const std::string&) key
  * @param (CollisionHandlerEventNode<T>*) collider_Ptr
  */
 template <typename T>
 inline void CollisionObject<T>::add_collider( const std::string& key, CollisionHandlerEventNode<T>* collider_Ptr )
 {
    nassertv( collider_Ptr != NULL );
    _colliders.add( key, collider_Ptr );
    
    // Tag the Collider as belonging to this Object
    collider_Ptr->set_tag( OBJECT_KEY_TAG, StringConversion::to_str( NodePath::get_key() ) ); // Note: NodePath::get_key() != key
 }
 
 /**
  * detach_collider() detaches the Collider matching the specified key
  * from its parent node, if the Collider has a parent node.
  *
  * @param (const std::string&) key
  */
 template <typename T>
 inline void CollisionObject<T>::detach_collider( const std::string& key )
 {
    _colliders.detach( key );
 }
 
 /**
  * detach_colliders() detaches all Colliders belonging to this Object
  * from their parents.
  */
 template <typename T>
 inline void CollisionObject<T>::detach_colliders()
 {
    _colliders.detach_all();
 }
 
 /**
  * get_collider() returns a pointer to the Collider matching the specified key.
  * If no Collider having the matching specified key is found, NULL is returned.
  *
  * @param (const std::string&) key
  * @return CollisionHandlerEventNode<T>*
  */
 template <typename T>
 inline CollisionHandlerEventNode<T>* CollisionObject<T>::get_collider( const std::string& key )
 {
    return static_cast<CollisionHandlerEventNode<T>*>( _colliders.get( key ) );
 }
 
 /**
  * get_first_active_collider_found() returns a pointer to the first active 
  * CollisionHandlerEventNode found contained by this CollisionObject.
  * If no CollisionHandlerEventNode is active, then a NULL pointer is returned.
  *
  * @param (const std::string&) key
  * @return CollisionHandlerEventNode<T>*
  */
 template <typename T>
 inline CollisionHandlerEventNode<T>* CollisionObject<T>::get_first_active_collider_found()
 {
    return _colliders.get_first_active_collider_found();
 }
 
 /**
  * remove_collider() removes the Collider matching the specified key
  * from the ColliderCollection belonging to this Object.
  *
  * @param (const std::string&) key
  */
 template <typename T>
 inline void CollisionObject<T>::remove_collider( const std::string& key )
 {
    _colliders.remove( key );
 }
 
 /**
  * remove_colliders() removes all DamageObjects from the DamageObjectCollection
  * belonging to this Object.
  */
 template <typename T>
 inline void CollisionObject<T>::remove_colliders()
 {
    _colliders.clear();
 }
 
 /**
  * reparent_collider_to() sets the collider of the specified key, contained by this CollisionObject, 
  * as a child of the specified NodePath.
  *
  * @param (const std::string&) key
  * @param (NodePath&) node_path
  */
 template <typename T>
 inline void CollisionObject<T>::reparent_collider_to( const std::string& key, NodePath& node_path )
 {
    _colliders.reparent_to( key, node_path );
 }
 
 /**
  * reparent_colliders_to() sets all colliders, contained by this CollisionObject, as children of the
  * specified NodePath.
  *
  * @param (NodePath&) node_path
  */
 template <typename T>
 inline void CollisionObject<T>::reparent_colliders_to( NodePath& node_path )
 {
    _colliders.reparent_to( node_path );
 }
 
 /**
  * tag_colliders() tags all colliders currently contained by this CollisionObject using the specified
  * tag_key with the specified tag_value.
  *
  * @param (const std::string&) tag_key
  * @param (const std::string&) tag_value
  */
 template <typename T>
 inline void CollisionObject<T>::tag_colliders( const std::string& tag_key, const std::string& tag_value )
 {
    ColliderCollection::iterator end_Itr( _colliders.end() );
    for ( ColliderCollection::iterator c_Itr( _colliders.begin() ); c_Itr != end_Itr; ++c_Itr )
    {
        c_Itr->set_tag( tag_key, tag_value );
    }
 }
 
 /**
  * turn_on_collision() makes this object capable of colliding.
  */
 template <typename T>
 inline void CollisionObject<T>::turn_on_collision()
 {
    // turn on all colliders
    _colliders.reparent_to( *this );
 }
 
 /**
  * turn_off_collision() makes this object incapable of colliding.
  */
 template <typename T>
 inline void CollisionObject<T>::turn_off_collision()
 {
    // turn off all colliders
    _colliders.detach_all();
 }
 
 /** PROTECTED FUNCTIONS **/
 
 /**
  * add_collider_without_tagging() adds the Collider specified by the collider_Ptr with the 
  * specified key into the ColliderCollection for this Object (without tagging the collider).
  *
  * @param (const std::string&) key
  * @param (CollisionHandlerEventNode<T>*) collider_Ptr
  */
 template <typename T>
 inline void CollisionObject<T>::add_collider_without_tagging( const std::string& key, CollisionHandlerEventNode<T>* collider_Ptr )
 {
    nassertv( collider_Ptr != NULL );
    _colliders.add( key, collider_Ptr );
 }
 
 /** FRIEND FUNCTIONS **/
 
 #endif // COLLISION_OBJECT_H